#!/usr/local/bin/perl
#
# This script implements a very simple REST server to serve content
# to our Zimlet. In production, SFU's Zimlets use an in-house REST 
# server with a back-end database, but it's not easily packagable to
# distribute with our Zimlets, so this script serves as a simple
# implementation that a site can use or use as a model to build
# their own in-house system.
#
# There are at least a thousand different ways to do this in Perl,
# most better than this one. If you like Perl, look into
# something like Mojolicious or Catalyst::Controller::Rest
# for a better way of doing it
#
# Note that SFU uses a 'ZSessionID' parameter to pass along a security
# token. The token ensures that the rquestor is an authenticated Zimbra
# user, but the token code is not portable outside of SFU and won't be
# supported in this code - i.e. all requests will be answered. This should be
# ok for GETs but is obviously undesirable for POSTs that could involve changing
# settings


use CGI;
use URI::Escape;
use JSON;

#$debug=1;
$basedir = "/home/hillman/pub_html/zimbra/rest";	# This is the base dir where REST content lives as files

$tipsdir = "$basedir/motd/tips";
$alertsdir = "$basedir/motd/alerts";
$dsdir = "$basedir/ds";

$q = new CGI;

$qtype = $q->request_method();		# GET or POST?
# Workaround - browsers can only send GET and POST but we may also want to DELETE:
$qtype = "DELETE" if ($q->param('rest_op') eq "delete");

$path = $q->path_info();		# The URL content after this script e.g. /cgi-bin/rest.cgi/some/other/stuff returns /some/other/stuff
$path =~ s/^\///;			# Remove leading slash from path

$path =~ s/\?.*//;			# If there are params, remove them from the path
@parts = split(/[\/?]/,$path);		# Break the REST string into its components
%params = $q->Vars;			# Put any trailing params after the REST string into a hash
if ($qtype eq "POST")			# Fetch POST data if present
{
	$postdata = $q->param('POSTDATA');
}

	# comment in for Trivial Security:
	# if ($qtype eq "POST" || $qtype eq "DELETE")
	# {
	#   if ($parts[0] ne "datastore" || $path =~ /^datastore\/motd\/global/)
	#   {
	#	# POSTing to a Restricted area. Verify IP is in range
	#	$addr = $ENV{"REMOTE_ADDR"}
	#	if ($addr =~ /123\.45\.67\.89/ || $addr =~ /123.45.68\./)
	#	{
	#		# Let through 123.45.67.89 and all of 123.45.68
	#	}
	#	else
	#	{
	#		error("403");
	#	}
	#    }
	# }
	# (you could also put the list of IPs in a file that only gets read
	#  if we need to verify IP. That way, the file can be modified without
	#  touching the cgi, which is constantly being accessed by end-users)

# Capture the REST filetype suffix, if there was one, to determine content-type we should return
if ($path =~ /\.([a-zA-Z]+)$/)
{
	$ctype = $1;
}
else
{
	# Assume text
	$ctype = "txt";
}

if ($debug)
{
	print_header("xml");
	print "path: ",join("/",@parts);
	print "\nquery type: $qtype\n";
	print "Post data: $postdata\n" if ($qtype eq "POST");
	read_tips();
	print "Loaded tips: $#tips\n";
	send_all_tips();
	exit 0;
}

if ($parts[0] eq "motd")
{
	# Process /motd/ REST services
	# We have two branches here - /tips and /pa
	if ($parts[1] eq "tips")
	{
		# Handle Tips of the Day
		#  REST calls supported:
		#   /motd/tips/random - returns a random tip
		#   /motd/tips/all.js - return all tips as a JSON object (used by admin app to edit tips)
		#   /motd/tips/<tip-id> - returns the specified tip
		#   /motd/tips/<tip-id/next - returns next tip in sequence
		#   /motd/tips/<tip-id>/previous - returns previous tip in sequence
		#
		$parts[2] =~ s/\..*$//;		# strip suffix

		# Do POSTs
		if ($qtype eq "POST")
		{
			# POST is used by the admin app to add or edit tips
			error() if ($parts[2] !~ /^\d+$/);	# Tip Id must be numeric
			post_file("$tipsdir/".$parts[2],$postdata);	# Write POST data out to tip file
			print_header($ctype);
			print "OK";
			exit 0;
		}
		elsif ($qtype eq "DELETE")	# Admin wants to delete a tip?
		{
			error() if (defined($parts[3]));
			unlink("$tipsdir/".$parts[2]) if (-e "$tipsdir/".$parts[2]);
			print_header($ctype);
			print "OK";
			exit 0;
		}
		elsif ($qtype ne "GET")		# error if it's not a GET at this point
		{
			error();
		}

		# Do GETs
		read_tips();		# Puts all tip IDs into @tips and as keys in %tips
		if ($parts[2] eq "random")
		{
			# Pick one at random
			$tip = int(rand($#tips+1));
			send_tip($tips[$tip]);
			exit 0;
		}
		elsif ($parts[2] eq "all")
		{
			send_all_tips();
			exit 0;
		}
		elsif (defined($tiphash{$parts[2]})) # was part-2 a valid Tip ID?
		{
			$tipnum = $tiphash{$parts[2]};
			if ($parts[3] eq "next")
			{
				$tipnum++;
			}
			elsif ($parts[3] eq "previous")
			{
				$tipnum--;
			}
			elsif (!defined($parts[3]))
			{
				# Do nothing - just causes the specified Tip ID to be returned
			}
			else
			{
				error();
			}
			$tipnum = 0 if ($tipnum > $#tips);
			$tipnum = $#tips if ($tipnum < 0);

			send_tip($tips[$tipnum]);
			exit 0;
		}
		else
		{
			error("204");		# Not an error, but no content avail
		}

		# End of Tips of the Day
	}
	elsif ($parts[1] eq "pa")
	{
		# Handle alerts. The following REST calls are supported: 
		#  /motd/pa/current.js - returns all alerts flagged as 'active'
		#  /motd/pa/all.js     - returns ALL alerts
		#  /motd/pa/<id>       - GET an ID or POST to an ID to change its content
		#
		$parts[2] =~ s/\..*$//;		# strip suffix
		if ($qtype eq "POST")
		{
			# POST is used by the admin app to add or edit tips
			$parts[2] =~ s/\..*$//;		# strip suffix
			error() if (defined($parts[3]));		# <id> must be the last element of the path
			error() if ($parts[2] !~ /^\d+$/);	# Id must be numeric
			post_file("$alertsdir/".$parts[2],$postdata);	# Write POST data out to tip file
			print_header($ctype);
			print "OK";
			exit 0;
		}
		elsif ($qtype eq "DELETE")	# Admin wants to delete an ID?
		{
			error() if (defined($parts[3]));
			unlink("$alertsdir/".$parts[2]) if (-e "$alertsdir/".$parts[2]);
			print_header($ctype);
			print "OK";
			exit 0;
		}
		elsif ($parts[2] eq "current" || $parts[2] eq "all")
		{
			send_alerts(($parts[2] eq "current") ? 1 : 0);
			exit 0;
		}
		elsif ($parts[2] =~ /^\d+$/)
		{
			if (-e "$alertsdir/".$parts[2])
			{
				print_header($ctype);
				print fetch_file("$alertsdir/".$parts[2]);
				exit 0;
			}
		}
		else
		{
			error("204");		# Not an error, but no content avail
		}
	}
	# Else no more methods in the motd branch so we fall through to 404 error

}
elsif ($parts[0] eq "datastore")
{
	# Process /datastore/ REST services
	# This is where we store global and user settings for Zimlets
	# use GET to query a setting and POST to change it
	#
	# The RIGHT way to do this would be one large mysql table with a field for the app ('motd'),
	# a field for the userid (a user or 'global') and fields for the variable name and value
	# but I'm not going to assume you're running a mysql db, so I'll just use the filesystem.
	# PLEASE feel free to rewrite this!

	if ($parts[1] ne "motd")
	{
		# All we need to do here is check whether parts[1] is a name
		# we want to trust. Since it's just a datastore, we could allow
		# any valid name here, but I'm just going to stick with only
		# allowing "motd"
		error("204");
	}
	else
	{
		# Currently supported REST calls
		# /motd/global.js - all global settings for Tips zimlet in JSON object
		# /motd/global/someVariable.js - get a particular variable
		# /motd/<userid>/showtip.txt - per-user variable for whether user has enabled tips or not
		# /motd/<userid>/lastseen - per-user last Tip-ID seen
		#
		# Note that this code doesn't really distinguish between "global" and 'userid'
		# It treats 'global' as just another user

		$p1 = $parts[1];
		$p2 = $parts[2];
		if (defined($parts[3]))
		{
			$p3 = $parts[3];
			$p3 =~ s/\..*$//;	# Strip suffix from last part. We captured it in 'ctype'
		}
		else
		{
			error() if ($qtype eq "POST");	# Can't post to all variables at once
			$p3 = undef;
			$p2 =~ s/\..*$//;
			# We can only return all variables as a json
			# object, so force response to be json
			$ctype="js";
		}
	
		if (! -d "$dsdir/$p1/$p2")
		{
			if ($qtype eq "POST")
			{
				# It's a POST, so create the directory
				mkdir("$dsdir/$p1/$p2");
			}
			else
			{
				# It's a GET to a non-existent dir. Return empty-set
				print_header($ctype);
				print_as_ctype(undef,$ctype);
				exit 0;
			}
		}

		# Good, directory exists
		if (defined($p3))
		{
			# Sanity check - if string is anything but alphanumerics, throw error
			error("500") if ($p3 =~ /\W/);
	
			if ($qtype eq "POST")
			{
				# Write to the file
				post_file("$dsdir/$p1/$p2/$p3",$postdata);
				print_header($ctype);
				print "OK";
				exit 0;
			}
			elsif ($qtype eq "GET")
			{
				# Return just one variable, if it exists
				if (-e "$dsdir/$p1/$p2/$p3")
				{
					$var = fetch_file("$dsdir/$p1/$p2/$p3");
					print_header($ctype);
					print_as_ctype($var,$ctype,$p3);
					exit 0;
				}
				else
				{
					error("204");		# Not an error, but no content avail
				}
			}
			# else fall through to error
		}
		else	# Requestor wants all variables for this object
		{
			# We only know how to return these as JSON (xml is ok, cuz that means no suffix was specified)
			error() if ($ctype ne "js" && $ctype ne "xml");	
			# Requestor wants all variables
			opendir(D,"$dsdir/$p1/$p2") or error("500");
			@vars = grep (/^[^\.]+$/, readdir(D) ); 
			closedir D;
			foreach $f (@vars)
			{
				$val = fetch_file("$dsdir/$p1/$p2/$f");
				$jsonhash{$f} = $val;
			}
			print_header($ctype);
			print encode_json(\%jsonhash);
			exit 0;
		}

	}

	# else fall through to 404 error
}
elsif ($parts[0] eq "isup")
{
	# Just returns 'OK'. Zimlet uses this to make sure the REST server is
	# functioning. If it isn't, the Zimlet disables itself
	print_header($ctype);
	print "OK";
	exit 0;
}

# Additional REST services would go here

# Fall through to a 404 error
error();

# Should never get here
exit 0;



# Read all tip IDs in from disk. Each tip is stored in a file in
# the $tipsdir directory. The file name is the tip ID. 
# To prevent potential abuse, keep IDs strictly numeric

sub read_tips()
{
	opendir(D,$tipsdir) or error("500");
	@tips = grep (/^\d+$/, readdir(D) ); # all-numeric filenames
	closedir(D);
	for($i = 0; $i <= $#tips; $i++)
	{
		$tiphash{$tips[$i]} = $i;
	}
}

# print a content-type header appropriate to whatever content-type
# the requestor asked for

sub print_header()
{
	$t = shift;
	# These are all the types we support right now
	#
	if ($t eq "js")
	{
		$type = "application/json";
	}
	elsif ($t eq "xml")
	{
		$type = "text/xml";
	}
	else
	{
		$type = "text/plain";
	}
	print $q->header(-type=>$type);
}

# Print a given variable in the appropriate format. We don't support
# many formats yet

sub print_as_ctype()
{
	my ($val,$ctype) = @_;

	if ($ctype eq "json")
	{
		print encode_json($val);
	}
	else
	{
		# Default - just print the value as text
		print $val;
	}
}

# Send a tip to the requestor.  This code uses files to store tips. The Tip file is created by a POST
# call from the Admin interface, so is in whatever format the admin interface created.
# Right now, that's an escaped querystring - e.g.
#  variable=value&variable=two%20words
#

sub send_tip()
{
	$tip_id = shift;
	error() if (! defined($tiphash{$tip_id}));	# Make sure tip exists

	print $q->header(-type=>'text/xml');
	print fetch_file("$tipsdir/$tip_id");
}


# Used by the Admin interface to fetch all tips as a JSON object
sub send_all_tips()
{
	my (%tipjson);
	foreach $t (@tips)
	{
		$msg = fetch_file("$tipsdir/$t");
		$tipjson{$t} = $msg;
	}
	print $q->header(-type=>'application/json');
	print encode_json(\%tipjson);
}


# send all alerts as a JSON object.  Takes one optional argument which, if defined,
# specifies that only alerts flagged as active should be returned
# The 'active' alerts REST call wants the data as a JSON array
# whereas the 'all' call wants it as a hash, so we assemble it as both. Sigh.

sub send_alerts($)
{
	# Read in any alerts first

	my (%alerthash,@alerts);
	my ($current) = shift;		# Only send active alerts?
	
	opendir(D,$alertsdir) or error("500");;
	@alertfiles = grep (/^\d+$/, readdir(D) ); # all-numeric filenames
	closedir(D);

	foreach $a (@alertfiles)
	{
		$msg = fetch_file("$alertsdir/$a");
		next if ($current && $msg !~ /messageStatus=active/);

		# Check message time boundaries
		if ($msg =~ /messageStartTime=(\d+)/)
		{
			$start = $1;
		}
		if ($msg =~ /messageEndTime=(\d+)/)
		{
			$end = $1;
		}
		if ($start && $end)
		{
			$now = time() * 1000;	# Time In ms.
			next if ($current && ($now < $start || $now > $end));
		}

		$alerthash{$a} = $msg;
		push (@alerts,$msg);
	}


	print $q->header(-type=>'application/json');
	print encode_json(($current) ? \@alerts : \%alerthash);
}
	

# Return the contents of a file as a string. To switch to using a database as your back-end
# store, replace this code with something appropriate

sub fetch_file($)
{
	$file = shift;

	open(F,$file) or error("500");
	$ret = join("",<F>);
	close F;
	return $ret;
}

# And this function writes POST data to disk as a file. To replace your back-end storage with
# a db, replace this code

sub post_file($$)
{
	my ($f,$data) = @_;

	open(F,">$f") or error("500");
	print F $data;
	close F;
}



sub error()
{
	$err = shift;
	if (defined($err))
	{
		print $q->header(-status=>$err);
	}
	else
	{
		print $q->header(-status=>'404 Not Found');
	}
	exit 0;
}
