/**
 * Copyright 2011 University of Guelph - Computing and Communication Services
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 * @overview
 * This zimlet adds a shortcut button to the task list
 * so the user can mark a task as completed without the 
 * need to open the task for edit.
 * 
 * @author Kennt Chan
 */
function ca_uoguelph_ccs_taskaddonHandlerObject () {
    // nothing
}
ca_uoguelph_ccs_taskaddonHandlerObject.prototype = new ZmZimletBase();
ca_uoguelph_ccs_taskaddonHandlerObject.prototype.constructor = ca_uoguelph_ccs_taskaddonHandlerObject;

function CcsTask(){}

CcsTask = ca_uoguelph_ccs_taskaddonHandlerObject;

CcsTask.OP_MARK_AS = "MARK_AS";
CcsTask.OP_MARK_AS_COMPLETE = "MARK_AS_COMPLETE";
CcsTask.OP_MARK_AS_PROGRESS = "MARK_AS_PROGRESS";
CcsTask.OP_MARK_AS_NOT_STARTED = "MARK_AS_NOT_STARTED";
CcsTask.OP_MARK_AS_WAITING = "MARK_AS_WAITING";
CcsTask.OP_MARK_AS_DEFERRED = "MARK_AS_DEFERRED";
CcsTask.SRC_TOOLBAR = "TOOLBAR";
CcsTask.SRC_CONTEXT_MENU = "CTX_MENU";
/**
 * zimlet init
 */
CcsTask.prototype.init = function() {
    this.overrideFunctions();
};

/**
 * Overrides the function that opens the pop-up so
 * this zimlet is notified when there is a right click on
 * a task item
 */
CcsTask.prototype.overrideFunctions = function() {
    if (typeof ZmTaskListController === "undefined") {
        AjxDispatcher.require("TasksCore");
    }
    
    ZmTaskListController.prototype._listActionListener = function(ev) {
        ZmListController.prototype._listActionListener.call(this, ev);
        var actionMenu = this.getActionMenu();
        appCtxt.notifyZimlets("onTaskMenuOpened", [this, actionMenu]);
        actionMenu.popup(0, ev.docX, ev.docY);
    };
};

/**
 * Creates a menu for the specified parent object and
 * according to the specified parameters.
 * @param params
 */
CcsTask.prototype.createMenu = function(params) {
    // add the menu options
    var btnMarkAs = params.parent.createOp(CcsTask.OP_MARK_AS, {
        text:   this.getMessage("mark_as"),
        index:  params.index,
        image:  "Task"
    });
    var menu = new ZmActionMenu({
        parent:     btnMarkAs,
        id:         btnMarkAs.getHTMLElId() + "|MENU",
        menuItems:  ZmOperation.NONE
    });
    
    var buttons = [{op: CcsTask.OP_MARK_AS_COMPLETE,    text:this.getMessage("mark_as_completed"),  image:"TaskViewCompleted"},
                   {op: CcsTask.OP_MARK_AS_NOT_STARTED, text:this.getMessage("mark_as_not_started"),image:"TaskViewNotStarted"},
                   {op: CcsTask.OP_MARK_AS_PROGRESS,    text:this.getMessage("mark_as_progress"),   image:"TaskViewInProgress"},
                   {op: CcsTask.OP_MARK_AS_WAITING,     text:this.getMessage("mark_as_waiting"),    image:"TaskViewWaiting"},
                   {op: CcsTask.OP_MARK_AS_DEFERRED,    text:this.getMessage("mark_as_deferred"),   image:"TaskViewDeferred"}];
    
    this.createButtons(buttons, menu, params.controller, params.src);
    btnMarkAs.setMenu(menu);
};

/**
 * Called when a toolbar is initialized, it add the "Mark as" button to the toolbar
 * @param app
 * @param toolbar
 * @param controller
 * @param viewId
 */
CcsTask.prototype.initializeToolbar = function(app, toolbar, controller, viewId) {
    if (viewId == ZmId.VIEW_TASKLIST) {
        var buttonIndex = 0;
        for (var i = 0; i < toolbar.opList.length; i++) {
                // get the index of the "Refresh" button
                if (toolbar.opList[i] === ZmOperation.EDIT) {
                        buttonIndex = i + 1;
                        break;
                }
        }
        // override the function to reset the operations in the toolbar as there is no method to
        // set when to enable or disable buttons based on the selection in the button api
        var originalFunction = controller._resetOperations;
        controller._resetOperations = function(parent, num) {
            originalFunction.apply(controller, arguments);
            parent.enable(CcsTask.OP_MARK_AS, num > 0);
        };
        
        this.createMenu({controller:controller, parent: toolbar, index:buttonIndex, src:CcsTask.SRC_TOOLBAR});
    }
};

/**
 * Called when the context menu is opened
 * @param controller
 * @param actionMenu
 */
CcsTask.prototype.onTaskMenuOpened = function(controller, actionMenu) {
    
    if (!actionMenu.getOp(CcsTask.OP_MARK_AS)) {
        actionMenu.createSeparator();
        
        this.createMenu({controller:controller, parent:actionMenu, src:CcsTask.SRC_CONTEXT_MENU});
        actionMenu.addPopupListener(new AjxListener(this, this.onRightClick, [controller, actionMenu]));
    }
};

/**
 * Creates each of the buttons for the menu
 * @param buttons button parameters
 * @param parent the context menu
 * @param controller
 * @param src
 */
CcsTask.prototype.createButtons = function(buttons, parent, controller, src) {
    
    for (var i = 0; i < buttons.length; i++) { 
    
        var params = buttons[i];
        var button = parent.createOp(params.op, {
            text:   params.text,
            image:  params.image
        });
        button.addSelectionListener(new AjxListener(this, this.markAsListener, [params.op, controller, src]));
    }
};

/**
 * Each item is marked as the selected operation and the task is saved.
 * @param controller
 */
CcsTask.prototype.markAsListener = function(op, controller, src) {
    var selected = CcsTask.SRC_TOOLBAR === src ? controller.getCurrentView().getSelection() : controller.getCurrentView().getDnDSelection();
    selected = (selected instanceof Array) ? selected : [selected];
    
    for (var i = 0; i < selected.length; i++) {
        var task = selected[i];
        task.viewMode = ZmCalItem.MODE_EDIT;
        
        switch (op) {
        case CcsTask.OP_MARK_AS_COMPLETE:
            task.pComplete = 100;
            task.status = ZmCalendarApp.STATUS_COMP;
            break;
        case CcsTask.OP_MARK_AS_PROGRESS:
            task.status = ZmCalendarApp.STATUS_INPR;
            break;
        case CcsTask.OP_MARK_AS_DEFERRED:
            task.status = ZmCalendarApp.STATUS_DEFR;
            break;
        case CcsTask.OP_MARK_AS_NOT_STARTED:
            task.pComplete = 0;
            task.status = ZmCalendarApp.STATUS_NEED;
            break;
        case CcsTask.OP_MARK_AS_WAITING:
            task.status = ZmCalendarApp.STATUS_WAIT;
            break;
        default:
            continue;
        }
        task.save();
    }    
    
};

/**
 * Changes the default behaviour of disabling the menu
 * option when there's more than one item selected.
 * @param controller ZmTaskListController 
 * @param actionMenu
 */
CcsTask.prototype.onRightClick = function(controller, actionMenu) {
    var selected = controller.getCurrentView().getDnDSelection();
    selected = (selected instanceof Array) ? selected : [selected];
    // default behaviour is to disable for more than one, changed here
    actionMenu.enable(CcsTask.OP_MARK_AS, selected.length > 0);    
};


