/**
 * Copyright 2011 University of Guelph - Computing and Communication Services
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 * @overview
 * This is the zimlet that finds duplicates in the contact list
 * It creates a toolbar menu button with the option of merging the selected contacts
 * or finding all the duplicates in the address book app.
 * 
 * @author Kennt Chan  
 */
function ca_uoguelph_ccs_contactcleanerplusHandlerObject() {
}
ca_uoguelph_ccs_contactcleanerplusHandlerObject.prototype = new ZmZimletBase();
ca_uoguelph_ccs_contactcleanerplusHandlerObject.prototype.constructor = ca_uoguelph_ccs_contactcleanerplusHandlerObject;

function CcsContactCleaner () {}

CcsContactCleaner = ca_uoguelph_ccs_contactcleanerplusHandlerObject;

CcsContactCleaner.prototype.init = function() {
};

/** Option to merge the displayed contacts */
CcsContactCleaner.prototype.MERGE = 1;
/** Option to keep only on of the contacts */
CcsContactCleaner.prototype.KEEP_ONE = 2;
/** Option to skip any processing to the contacts */
CcsContactCleaner.prototype.DO_NOTHING = 3;
/** Limit of contacts to query at a time */
CcsContactCleaner.prototype.LIMIT = 500;
/** Toolbar button op id */
CcsContactCleaner.TOOLBAR_OPERATION = "MORE_ACTIONS";
/** Menu items */
CcsContactCleaner.prototype.OPTION_MERGE = "ccMenuItemMergeContacts";
CcsContactCleaner.prototype.OPTION_FIND_DUPS = "ccMenuItemFindDups";

/**
 * Create the main container for the zimlet
 */
CcsContactCleaner.prototype.initializeEmptyDlg = function() {
    var parentView = new DwtComposite(this.getShell());
    parentView.getHtmlElement().style.overflow = "auto";
    return parentView;
};

/**
 * Creates the dialog window 
 */
CcsContactCleaner.prototype.createContactCleanerDialog = function() {
    if (this.contactCleanerWindow) return;  
    // define the buttons
    this.closeButtonId = Dwt.getNextId();
    var closeButton = new DwtDialog_ButtonDescriptor(this.closeButtonId, this.getMessage("close_button"), DwtDialog.ALIGN_RIGHT);
    this.mergeAllButtonId = Dwt.getNextId();
    var mergeAllButton = new DwtDialog_ButtonDescriptor(this.mergeAllButtonId, this.getMessage("merge_all_button"), DwtDialog.ALIGN_LEFT);
    
    // create dialog window
    this.contactCleanerWindow = this._createDialog({title:this.getMessage("cleaner_dialog_title"), view:this.containerView, standardButtons : [DwtDialog.NO_BUTTONS], extraButtons:[mergeAllButton, closeButton]});
    this.contactCleanerWindow.setButtonListener(this.mergeAllButtonId, new AjxListener(this, this.mergeAllBtnListener));
    this.contactCleanerWindow.setButtonListener(this.closeButtonId, new AjxListener(this, this.closeWindowBtnListener));
};

/**
 * Creates theb basic html layout
 * @returns the string with the html code
 */
CcsContactCleaner.prototype.constructContactManagerView = function() {
    var i = 0;
    var html = [];
    html[i++] = "<div id='ccBaseContainerDiv'>";
    // div containing the wait widget.
    html[i++] = "<div id='ccScanWait' class='ccScanWaitStyle'><span id='waitingMessage'>";
    html[i++] = this.getMessage("scanning");
    html[i++] = "</span><img id='busyimg' src=\"" + this.getResource("cc_busy.gif") + "\"/>";
    html[i++] = "</div>";
       
    // div containing the options for a duplicate
    html[i++] = "<div id='ccSingleDuplicateContainer' class='ccSingleDuplicateCont'>";
    html[i++] = this.getMessage("select_info");
    html[i++] = "<br>";
    html[i++] = "<div id='ccSingleDuplicateOptions'>";
    html[i++] = "</div>";
    html[i++] = "<br>";
    html[i++] = "<div class='ccNumberInfo'>"; //contains label on number of duplicates
    html[i++] = "<span id='ccSpanCurrent'></span>&nbsp;";
    html[i++] = this.getMessage("of");
    html[i++] = "&nbsp;<span id='ccSpanTotal'></span>&nbsp;";
    html[i++] = this.getMessage("of_duplicates");
    html[i++] = "</div>";
    html[i++] = "</div>";
    html[i++] = "</div>";
    html[i++] = "";
    return html.join("");
};

/**
 * Clears the html element of any child nodes
 */
CcsContactCleaner.prototype.clearChildNodes = function(container) {
    if (container.hasChildNodes()){
        while ( container.childNodes.length >= 1 ) {
            container.removeChild(container.firstChild);
        }
    }
};

/**
 * Resizes the main dialog
 * @param w the width
 * @param h the height
 */
CcsContactCleaner.prototype.resizeMainContainer = function(w,h) {
    this.containerView.setSize(w,h);
};

/**
 * Resets the ui 
 */
CcsContactCleaner.prototype.resetView = function() {
    
    //this.contactCleanerWindow.setButtonEnabled(this.nextButtonId, false);
    this.contactCleanerWindow.setButtonEnabled(this.mergeAllButtonId, false);    
    this.contactCleanerWindow.setButtonEnabled(this.closeButtonId, true);
    this.contactCleanerWindow.getButton(this.closeButtonId).setText(this.getMessage("close_button"));
    this.requestCount = 0;
    
    this.containerView.setSize(350, 60);
    document.getElementById("ccScanWait").style.display = "";
    document.getElementById('busyimg').style.display = "";
    document.getElementById("waitingMessage").innerHTML = this.getMessage("scanning");
    document.getElementById("ccSingleDuplicateContainer").style.display = "none";
    this.clearChildNodes(document.getElementById("ccSingleDuplicateOptions"));
};

/**
 * Show the waiting dialog when a request is pending.
 */
CcsContactCleaner.prototype.showWaitingDialog = function() {
    if (this.requestCount > 0) {
        this.contactCleanerWindow.setButtonEnabled(this.mergeAllButtonId, false);
        this.contactCleanerWindow.setButtonEnabled(this.closeButtonId, false);
        
        document.getElementById("waitingMessage").innerHTML = this.getMessage("processing");
        document.getElementById("ccSingleDuplicateContainer").style.display = "none";
        this.resizeMainContainer(350, 60);
        document.getElementById("ccScanWait").style.display = "";
        this.waitForRequestAndClose();
    } else {
        this.waitForRequestAndClose();
    }
};

/**
 * Waits for any request to be completed and closes the dialog
 */
CcsContactCleaner.prototype.waitForRequestAndClose = function() {
    if (this.requestCount > 0) {
        var self = this;
        setTimeout(function() { self.waitForRequestAndClose();}, 100);
    } else {
        this.close();
    }
};

/**
 * Closes the dialog
 */
CcsContactCleaner.prototype.close = function() {
    this.iterator = null;
    this.duplicateList = null;
    this.contactCleanerWindow.popdown();
    // refresh the current query as the contact action may have not updated the ui completely.
    try {
        var q = appCtxt.getSearchController().currentSearch.query;
        appCtxt.getSearchController().search({query:q});
    } catch(e) {
    }
};

/**
 * Takes the list of address books and joins them into a query
 * @returns string
 */
CcsContactCleaner.prototype.getContactFolders = function() {
    var folderList = appCtxt.getFolderTree().asList();
    var numfolders = folderList.length;
    var contactFolders = [];
    for (var i = 0; i < numfolders; i++) {
        var folder = folderList[i];
        if (folder.type === ZmOrganizer.ADDRBOOK) {
            contactFolders.push(folder.createQuery());
        }
    }
    return contactFolders.join(" OR ");
};

/**
 * Gets the contact list from the server
 * @param offset the position to start from in the query
 * @param contactList the array containing the contacts requested previously
 */
CcsContactCleaner.prototype.getContacts = function(offset, contactList) {
    // create the json object for the search request
    var jsonObj = {SearchRequest:{_jsns:"urn:zimbraMail"}};
    var request = jsonObj.SearchRequest;
    request.sortBy = ZmSearch.NAME_ASC;
    ZmTimezone.set(request, AjxTimezone.DEFAULT);
    request.locale = { _content: AjxEnv.DEFAULT_LOCALE };
    request.offset = 0;
    request.types = ZmSearch.TYPE[ZmItem.CONTACT];
    request.query = this.getContactFolders();
    request.offset = offset || 0;
    request.limit = this.LIMIT;
    
    contactList = contactList || [];
    var searchParams = {
            jsonObj:jsonObj,
            asyncMode:true,
            callback:new AjxCallback(this, this.handleGetContactsResponse, [contactList]),
            errorCallback:new AjxCallback(this, this.handleGetContactsError)
    };
    appCtxt.getAppController().sendRequest(searchParams);
};

/**
 * Parses the response from the servers, if there are still some contacts
 * to be retrieved call the getContacts function again.
 * @param contactList the current list of contacts
 * @param result the result object
 */
CcsContactCleaner.prototype.handleGetContactsResponse = function(contactList, result) {
    if (result) {
        var response = result.getResponse().SearchResponse;        
        var responseContactList = response[ZmList.NODE[ZmItem.CONTACT]];
        if (responseContactList) {
            var numContacts = responseContactList.length;
            for (var i = 0; i < numContacts; i++) {
                contactList.push(responseContactList[i]);
            }
        }
        if (response.more) {
            this.getContacts(response.offset + this.LIMIT, contactList);
        } else {
            this.parseContactList(contactList);
        }
    }
};

/**
 * Show message in case of error.
 * @param result
 */
CcsContactCleaner.prototype.handleGetContactsError = function(result) {
    document.getElementById('busyimg').style.display = 'none';
    document.getElementById("waitingMessage").innerHTML = this.getMessage("error_get_contacts");
};

/**
 * Open the dialog window for this zimlet
 */
CcsContactCleaner.prototype.openCleanerDialog = function() {
     
    if (!this.contactCleanerWindow) {
        this.containerView = this.initializeEmptyDlg();
        this.containerView.setScrollStyle(Dwt.SCROLL);
        this.containerView.getHtmlElement().innerHTML = this.constructContactManagerView();
        this.createContactCleanerDialog();        
    }    
    this.resetView();
    this.contactCleanerWindow.popup();
    
    AjxDispatcher.run("GetContactController");
    this.getContacts();
};

/**
 * Creates the selection options for each duplicate case.
 * @param duplicate the object that contains info on the duplicate and the related contacts
 * @param currentItemNumber number, indicates the current position in the array of duplicates
 */
CcsContactCleaner.prototype.buildContactOptions = function (duplicate, currentItemNumber) {   
    if (duplicate) {
        var contacts = duplicate.getContacts();
        
        var parentContainer = document.getElementById("ccSingleDuplicateOptions");
        this.clearChildNodes(parentContainer);
        
        var container = document.createElement("div");
        container.className = "optionsContainer";
        container.appendChild(document.createElement("br"));
        for (var i = 0; i < contacts.length; i++) {
            
            var contact = contacts[i];
            
            var keepButton = new DwtButton(this.getShell());
            keepButton.setText(this.buildButtonText(contact));
            keepButton.setToolTipContent(this.buildPreview(contact));
            keepButton.addSelectionListener(new AjxListener(this, this.applyActionButtonListener, {action:this.KEEP_ONE, dup:duplicate, contactId:contact.id}));
            keepButton.setSize(330, 35);
            keepButton.getHtmlElement().className = keepButton.getHtmlElement().className + " optionsContainer"; 
            keepButton.getHtmlElement().firstChild.style.width = "100%";
            document.getElementById(keepButton.getHTMLElId() + "_title").style.height = "30px";
            container.appendChild(keepButton.getHtmlElement());
        }
        
        var mergeButton = new DwtButton(this.getShell());
        mergeButton.setText(this.getMessage("merge"));
        mergeButton.setToolTipContent(this.getMessage("merge_tooltip"));
        mergeButton.addSelectionListener(new AjxListener(this, this.applyActionButtonListener, {action:this.MERGE, dup:duplicate}));
        mergeButton.setSize(330, 35);
        mergeButton.getHtmlElement().className = mergeButton.getHtmlElement().className + " optionsContainer"; 
        mergeButton.getHtmlElement().firstChild.style.width = "100%";
        document.getElementById(mergeButton.getHTMLElId() + "_title").style.height = "30px";
        container.appendChild(mergeButton.getHtmlElement());
        
        var doNothingButton = new DwtButton(this.getShell());
        doNothingButton.setText(this.getMessage("do_nothing"));
        doNothingButton.setToolTipContent(this.getMessage("skip_tooltip"));
        doNothingButton.addSelectionListener(new AjxListener(this, this.applyActionButtonListener, {action:this.DO_NOTHING}));
        doNothingButton.setSize(330, 35);
        doNothingButton.getHtmlElement().className = doNothingButton.getHtmlElement().className + " optionsContainer"; 
        doNothingButton.getHtmlElement().firstChild.style.width = "100%";
        document.getElementById(doNothingButton.getHTMLElId() + "_title").style.height = "30px";
        container.appendChild(doNothingButton.getHtmlElement());
        parentContainer.appendChild(container);
        document.getElementById("ccSpanCurrent").innerHTML = currentItemNumber;
    }
    
};

/**
 * Creates the text to be displayed in the "keep" buttons
 * @param contact ZmContact that contains the info to be used in the button label
 */
CcsContactCleaner.prototype.buildButtonText = function(contact) {
    var text = [];
    var i = 0;
    var attr = contact.attr || contact._attrs;
    text[i++] = this.getMessage("use");
    text[i++] = " \"";
    text[i++] = (contact.getFileAs) ? contact.getFileAs() : contact.fileAsStr;
    text[i++] = "\" <";
    text[i++] = attr[ZmContact.F_email];
    text[i++] = ">";
    text = text.join("");
    return AjxStringUtil.convertToHtml(text.length > 50 ? text.substring(0, 50) + "..." : text);
};

/**
 * Creates the html content for each row of details, shows name of the attribute and value.
 * @param html array that contains the html code
 * @param i the current index for the html array
 * @param attr the attribute object from the contact
 * @param attrNames the names of the attributes to be processed
 * @param ignoreHash attribute names to be ignore from the preview
 * @returns the new position of the index.
 */
CcsContactCleaner.prototype.buildPreviewRows = function(html, i, attr, attrNames, ignoreHash) {
    var numNames = attrNames.length;
    ignoreHash = ignoreHash || {};
    for (var j = 0; j < numNames; j++) {
        var attrname = attrNames[j];
        var attrvalue = attr[attrname];
        if (!ignoreHash[attrname.replace(/\d+$/,"")] && attrvalue) {
            html[i++] = "<tr><td><b>";
            html[i++] = ZmContact._AB_FIELD[attrname] || attrname;
            html[i++] = "</b></td><td>";
            html[i++] = attrvalue;
            html[i++] = "</td></tr>";
        }
    }
    return i;
};

/**
 * Creates the html content to be shown in each button as a preview of the 
 * contact information.
 * @param contact the ZmContact info to show
 * @returns the string representation of the html code to show.
 */
CcsContactCleaner.prototype.buildPreview = function(contact) {
    var html = [];
    var i = 0;
    var attr = contact.attr || contact._attrs;
    html[i++] = "<table><tbody>";
    // set the main info
    i = this.buildPreviewRows(html, i, attr, [ZmContact.F_firstName, ZmContact.F_lastName]);
    // set the email info
    i = this.buildPreviewRows(html, i, attr, contact.emailList, {ccSimpleFullName:true});
    // add the rest of the attributes
    i = this.buildPreviewRows(html, i, attr, Object.keys(attr), {firstName:true, lastName:true, email:true, notes:true, ccSimpleFullName:true, firstLast:true});
    i = this.buildPreviewRows(html, i, attr, [ZmContact.F_notes]);
    html[i++] = "</tbody></table>";
    html[i++] = "";
    return html.join("");
};

/**
 * Sets the interface to show each one of the duplicates.
 */
CcsContactCleaner.prototype.populateContactRows = function () {
    
    this.iterator = new DuplicateIterator(this.duplicateList.getDuplicatesArray());
    this.buildContactOptions(this.iterator.next(), this.iterator.currentItemNumber());
    document.getElementById("ccSpanTotal").innerHTML = this.iterator.getSize();
    this.contactCleanerWindow.getButton(this.closeButtonId).setText(this.getMessage("skip_remaining_button"));
    // resize the main window
    document.getElementById("ccScanWait").style.display = "none";
    this.resizeMainContainer(400, 215);
    document.getElementById("ccSingleDuplicateContainer").style.display = "";
    
    //enable controls
    this.contactCleanerWindow.setButtonEnabled(this.mergeAllButtonId, true);
};

/**
 * Moves onto the next contact to be processed.
 */
CcsContactCleaner.prototype.nextContact = function() {
    if (this.iterator.hasNext()) {
        this.buildContactOptions(this.iterator.next(), this.iterator.currentItemNumber());
    } else {
        this.showWaitingDialog();
    }
};

/**
 * Displays the message in case there are no duplicates found.
 */
CcsContactCleaner.prototype.showNoContacts = function() {
    document.getElementById('busyimg').style.display = 'none';
    document.getElementById("waitingMessage").innerHTML = this.getMessage("no_duplicates");
};

/**
 * Duplicate contact object.
 * Contains the information of which contacts are part of this duplicate.
 */
function DupContact(id, contact) {
    this.ids = [id];
    // the duplicate contacts for this entry
    this.contacts = [];
    this.contactIds = [];
    this.addContact(contact);
}

/**
 * Add the contact to the duplicate only if it hasn't been added before.
 * @param contact the ZmContact
 */
DupContact.prototype.addContact = function(contact) {
    if (contact && !this.contactIds[contact.id]) {
        this.contacts.push(contact);
        this.contactIds[contact.id] = true;
    }
};

/**
 * Merges the information of two duplicate object onto one.
 * @param duplicate the duplicate to merge into.
 */
DupContact.prototype.merge = function(duplicate) {
    var dupIds = duplicate.getIds();
    for (var i = 0; i < dupIds.length; i++) {
        this.ids.push(dupIds[i]);
    }   
    var contacts = duplicate.getContacts();
    var size = contacts.length;
    for (var j = 0; j < size; j++) {
        var contactToMerge = contacts[j];
        if (!this.contactIds[contactToMerge.id]) {
            this.addContact(contactToMerge);
        }
    }
};

/**
 * Gets the number of contacts for this duplicate.
 * @returns number
 */
DupContact.prototype.size = function() {
    return this.contacts.length;
};

/**
 * Returns the duplicate id.
 * @returns {Array}
 */
DupContact.prototype.getIds = function() {
    return this.ids;
};

/**
 * Returns the array of contacts.
 * @returns {Array}
 */
DupContact.prototype.getContacts = function() {
    return this.contacts;
};

/**
 * A list that contains all the information on duplicates.
 */
function DuplicateList() {
    // hash of the emails that contain a reference to the duplicate id
    this.emailHash = [];
    // hash of the contact ids that contain a reference to the duplicate
    this.contactHash = [];
    
    this.duplicates = [];
    this.duplicateId = 0;
}

DuplicateList.CC_fullName = "ccSimpleFullName";
/**
 * Add the contact to the respective duplicate item.
 * @param contact ZmContact
 */
DuplicateList.prototype.add = function(contact) {
    // use attr if the object was initialized as ZmContact, _attrs otherwise
    var attr = contact.attr || contact._attrs;
    
    // this fixes a problem with some leftover information in group from a previous version
    if (attr[ZmContact.F_type] === "group") {
        // ignore
        return;
    }
    
    // use to make a simple comparison using first and last name
    attr[DuplicateList.CC_fullName] = (attr[ZmContact.F_firstName] || "") + (attr[ZmContact.F_lastName] ? (" " + attr[ZmContact.F_lastName]) : "");
    
    // use to check all the email addresses
    contact.emailList = [];
    for (var name in attr) {
        if (attr.hasOwnProperty(name)) {
            if (ZmContact.F_email === name.replace(/\d+$/,"")) {
                contact.emailList.push(name);
            }
        }
    }
    contact.emailList.sort();
    // add the custom full name to detect full name duplicates
    contact.emailList.push(DuplicateList.CC_fullName);
    var numEmails = contact.emailList.length;
    
    for (var i = 0; i < numEmails; i++) {
        var email = attr[contact.emailList[i]];
        if (email) {
            email = email.toLowerCase();
            if (this.emailHash.hasOwnProperty(email)) {
    
                // add the contact to the duplicate
                var duplicate = this.duplicates[this.emailHash[email]];
                            
                duplicate.addContact(contact);
                
                var duplicateFromContactHash = this.contactHash[contact.id];
                // if there is no duplicate associated with this contact, add
                if (!duplicateFromContactHash) {
                    this.contactHash[contact.id] = duplicate;       
                } else if (duplicateFromContactHash != duplicate) {
                    // if the duplicate entries are different, merge
                    duplicate.merge(duplicateFromContactHash);
                    this.contactHash[contact.id] = duplicate;
                    var idsToRefresh = duplicate.getIds();
                    for (var j = 0; j < idsToRefresh.length; j++) {
                        this.duplicates[idsToRefresh[j]] = duplicate;
                    }
                }
            } else if (this.contactHash[contact.id]) {
                // if the contact was already processed, add the reference to the email
                var dup = this.contactHash[contact.id];
                this.emailHash[email] = dup.getIds()[0];
            } else {
                // create a new duplicate object for this contact.
                var duplicate = new DupContact(this.duplicateId, contact);
                this.contactHash[contact.id] = duplicate;
                this.duplicates.push(duplicate);
                this.emailHash[email] = this.duplicateId;
                this.duplicateId = this.duplicates.length;
            }
        }
    }
};

/**
 * This function will remove all the DupContact object that have only one contact
 * associated as that would mean that there are no duplicates for them, as well
 * as the duplicates that are references more than once.
 */
DuplicateList.prototype.cleanNonDuplicates = function() {
    this.emailHash = null;
    this.contactHash = null;
    var size = this.duplicates.length;
    var duplicatesOnlyArray = [];
    for (var i = 0; i < size; i++) {
        var duplicate = this.duplicates[i];
        if (duplicate) {        
            // remove duplicate entries as well
            var duplicatesIds = duplicate.getIds();
            for (var j = 1; j < duplicatesIds.length; j++) {
                this.duplicates[duplicatesIds[j]] = null;
            }
            
            // keep only if it has more that one contact
            if (duplicate.size() > 1){
                duplicatesOnlyArray.push(duplicate);
            }
        }
    }
    // ?? order by name?
    this.duplicates = duplicatesOnlyArray;
};

/**
 * Returns the duplicates
 */
DuplicateList.prototype.getDuplicatesArray = function() {
    return this.duplicates;
};

/**
 * Whether the duplicate list contains any element.
 * @returns {Boolean}
 */
DuplicateList.prototype.isEmpty = function() {
    return this.duplicates.length === 0;
};

/**
 * This is the callback function after receiving the contact list, which it uses to create the duplicate list.
 * @param contactList ZmContactList object that contains all the contacts, if any changes were made to the
 * contact list after the last mailbox refresh, the contact list may be outdated.
 */
CcsContactCleaner.prototype.parseContactList = function (contactList) {
    if (!contactList) return;
    
    var size = contactList.length;
    this.duplicateList = new DuplicateList();
    
    for (var i = 0; i < size; i++) {
        var contact = contactList[i];
        this.duplicateList.add(contact);
    }
    this.duplicateList.cleanNonDuplicates();
    
    if (!this.duplicateList.isEmpty()) {
        this.populateContactRows();
        //this.contactList = contactList;
    } else {
        this.showNoContacts();
    }
};

/**
 * Creates the request xml info on moving the duplicates to trash and creating a new Contact
 * using the params provided.
 * @param params contain soapDoc    : the soap request doc.
 *                       moveToTrash: the array with the contact ids to move.
 *                       newAttrs   : the hash that contains all the attributes for the new contact.
 */
CcsContactCleaner.prototype.createRequest = function(params) {
    if (!params) {
        return;
    }
    
    // create SOAP request doc
    var soapDoc = params.soapDoc;
    // move the duplicates to trash
    var moveToTrashIds = params.moveToTrash;
    if (moveToTrashIds) {
        var contactActionReq = soapDoc.set("ContactActionRequest", null, null, "urn:zimbraMail");
        var action = soapDoc.set("action");
        action.setAttribute("op", "move");
        action.setAttribute("l", ZmFolder.ID_TRASH);
        action.setAttribute("id", moveToTrashIds.join(","));
        contactActionReq.appendChild(action);
    }
    
    // modify the contact if it was merged.
    var newAttrs = params.newAttrs;
    if (newAttrs) {
        var modifyContactReq = soapDoc.set("CreateContactRequest", null, null, "urn:zimbraMail");
        var doc = soapDoc.getDoc();
        var cn = doc.createElement("cn");
        //cn.setAttribute("id", contactToModify.id);
                        
        for (var name in newAttrs) {
            if (name == ZmContact.F_folderId)
                continue;
            var a = soapDoc.set("a", newAttrs[name], cn);
            a.setAttribute("n", name);
        }
        modifyContactReq.appendChild(cn);
    }
};

/**
 * Compares each one of the attributes of the two attribute hashes passed.
 * @param referenceAttrs the hash containing the original attributes.
 * @param compareAttrs the hash containing the attributes to compare.
 */
CcsContactCleaner.prototype.compareAttributes = function(referenceAttrs, compareAttrs) {
    // assuming a reasonably large suffix for merged attribute names 
    var attrId = 100;
    // use this to make the unique email item list and keep order
    var emailset = new EmailSet();
    
    for (var name in compareAttrs) {
        // the folder attribute is ignored as well as the attributes that are in the ignore list
        if (compareAttrs.hasOwnProperty(name) && name !== ZmContact.F_folderId && name !== DuplicateList.CC_fullName && !ZmContact.IS_IGNORE[name]) {
            var origAttr = referenceAttrs[name];
            var compareAttr = compareAttrs[name];
            // if there is no attribute in the original
            if (!origAttr && compareAttr) {
                referenceAttrs[name] = compareAttr;
            
            } else if (origAttr !== compareAttr) {
                var namePrefix = name.replace(/\d+$/,"");
                if (namePrefix === ZmContact.F_email) {
                    // emails should be unique, use this set to check
                    emailset.add(compareAttr);
                } else if (name === ZmContact.F_notes) {
                    // in the case of notes, append them
                    referenceAttrs[name] = origAttr + ". " + compareAttr;
                }
                // ignore some of the attributes that only have one possible value
                // i.e. don't follow phone, phone2, phone3
                else if (name !== ZmContact.F_firstName
                      && name !== ZmContact.F_lastName
                      && name !== ZmContact.F_middleName
                      && name !== ZmContact.F_middleName
                      && name !== ZmContact.F_company) {
                    // remove the suffix number if any
                    referenceAttrs[namePrefix+(attrId++)] = compareAttr;
                }
            }
        }
    }
    // add the email from the reference to the set
    for (var attrName in referenceAttrs) {
        if (referenceAttrs.hasOwnProperty(attrName) && ZmContact.F_email === attrName.replace(/\d+$/,"")) {
            emailset.add(referenceAttrs[attrName]);
            delete referenceAttrs[attrName];
        }
    }
    // add the emails to the attributes
    for (var i = 0; i < emailset.emails.length; i++) {
        referenceAttrs[ZmContact.F_email + i] = emailset.emails[i];
    }
};

/**
 * Merges the duplicates into a new contact and moves the duplicates to the trash. 
 * @param params
 */
CcsContactCleaner.prototype.cleanUpDuplicate = function(params) {
    var duplicate = params.duplicate;
    if (duplicate) {
        var contacts = duplicate.getContacts();        
        var merge = params.merge;
        var mergedAttrs = {};
        var mergedTags = params.mergedTags;
        var contactToKeepId = params.contactToKeep || null;
        // move the rest
        var contactsToDelete = [];
        for (var i = 0; i < contacts.length; i++) {
            var contactToCompare = contacts[i];
            if (merge) {
                // compare attributes.
                var attrs = contactToCompare.attr || contactToCompare._attrs;
                this.compareAttributes(mergedAttrs, attrs);
            }
            if (contactToCompare.id !== contactToKeepId) {
                contactsToDelete.push(contactToCompare.id);
            }
            // keep tags
            mergedTags.add(contactToCompare.tags || contactToCompare.t);
        }
        // reorder the attributes
        if (merge) {
            mergedAttrs = ZmContact.getNormalizedAttrs(mergedAttrs);
        }
        
        this.createRequest({
            newAttrs : mergedAttrs,
            moveToTrash: contactsToDelete,
            soapDoc: params.soapDoc
        });
    }
};

/**
 * Add the tag information to the list to be processed later.
 * @param taglist the array containing the tags sets
 * @param tagset to be added.
 * @param index the index of the contact in the request.
 */
CcsContactCleaner.prototype.addTags = function (taglist, tagset, index){
    if (tagset.size > 0) {
        taglist.push({tagset:tagset, index:index});
    }
    return index++;
};

/**
 * Creates the batch request for cleaning up the duplicates.
 * 
 */
CcsContactCleaner.prototype.cleanUp = function(params) {
    // create SOAP request doc
    var soapDoc = AjxSoapDoc.create("BatchRequest", "urn:zimbra");
    soapDoc.setMethodAttribute("onerror", "continue");
    var taglist = [];
    var mergedTags = new TagSet();
    var index = 0;
    // merge each duplicate
    if (params.mergeRemaining) {
        // process the one that is currently shown
        var currentDuplicate = this.iterator.current();
        this.cleanUpDuplicate({duplicate:currentDuplicate, merge:true, soapDoc: soapDoc, mergedTags: mergedTags});
        index = this.addTags(taglist, mergedTags, index);
        // process the remaining duplicates
        while (this.iterator.hasNext()) {
            mergedTags = new TagSet();
            currentDuplicate = this.iterator.next();            
            this.cleanUpDuplicate({duplicate:currentDuplicate, merge:true, soapDoc: soapDoc, mergedTags: mergedTags});
            index = this.addTags(taglist, mergedTags, index);
        }
    } else {        
        this.cleanUpDuplicate({duplicate: params.duplicate, contactToKeep: params.contactToKeep, merge: params.merge, soapDoc: soapDoc, mergedTags: mergedTags});
        this.addTags(taglist, mergedTags, index);
    }
    
    // send the batch request to the server
    var respCallback = new AjxCallback(this, this.contactModificationCallbackHandler, [taglist]);
    appCtxt.getAppController().sendRequest({soapDoc:soapDoc, asyncMode:true, callback:respCallback});
    this.requestCount++;
};

/**
 * Creates a batch request for tagging the contacts.
 * @param tags the array containing the tags sets
 * @param rContactList the response array containing the info on the contacts.
 */
CcsContactCleaner.prototype.createTaggingRequest = function(tags, rContactList) { 
  if (tags && rContactList) {
      
      // create SOAP request doc
      var soapDoc = AjxSoapDoc.create("BatchRequest", "urn:zimbra");
      soapDoc.setMethodAttribute("onerror", "continue");
      
      var numTags = tags.length;
      
      for ( var i = 0; i < tags.length; i++) {
          var tag = tags[i];
          var respcn = rContactList[tag.index];
          if (respcn) {
              var contact = respcn.cn[0];
              var tagHash = tag.tagset.hash;
              for (var tagId in tagHash) {
                  if (tagHash.hasOwnProperty(tagId)) {
                      var contactActionReq = soapDoc.set("ContactActionRequest", null, null, "urn:zimbraMail");
                      var action = soapDoc.set("action");
                      action.setAttribute("op", "tag");
                      action.setAttribute("id", contact.id);
                      action.setAttribute("tag", tagId);
                      contactActionReq.appendChild(action);
                  }
              }
          }
      }
      
      // send the  request to the server
      var respCallback = new AjxCallback(this, this.taggingCallbackHandler);
      appCtxt.getAppController().sendRequest({soapDoc:soapDoc, asyncMode:true, callback:respCallback});
      this.requestCount++;
  }
};

/**
 * Callback function for the tagging request.
 * @param result
 */
CcsContactCleaner.prototype.taggingCallbackHandler = function(result) {
    this.requestCount--;
};

/**
 * Callback function for the contact modification request.
 * @param args and array of the tags to be set.
 * @param result
 */
CcsContactCleaner.prototype.contactModificationCallbackHandler = function(args, result) {
    this.requestCount--;
    var rContactList = result.getResponse().BatchResponse.CreateContactResponse;
    this.createTaggingRequest(args, rContactList);
};

/**
 * This method gets called by the Zimlet framework when a toolbar is created.
 *
 * @param {ZmApp} app
 * @param {ZmButtonToolBar} toolbar
 * @param {ZmController} controller
 * @param {String} viewId
 * 
 */
CcsContactCleaner.prototype.initializeToolbar = function(app, toolbar, controller, viewId) {

    // check if the button is already present, as of version 6.09 this function is called twice when
    // initialiazing the toolbar
    if (viewId == ZmId.VIEW_CONTACT_SIMPLE && !toolbar.getButton(CcsContactCleaner.TOOLBAR_OPERATION)) {      
        var buttonParams = {
            text : this.getMessage("btn_more_actions"),
            tooltip : this.getMessage("btn_more_actions_tooltip"),
            index: toolbar.opList.length // position button before the text item in the toolbar
        };
        
        // creates the button with an id and params containing the button 
        var button = toolbar.createOp(CcsContactCleaner.TOOLBAR_OPERATION, buttonParams);
        var menu = new ZmPopupMenu(button, null, button.getHTMLElId() + "|MENU");
        
        // add the menu items
        menu.createMenuItem(this.OPTION_MERGE, {text: this.getMessage("mi_merge_contacts"), image: "TaskNormal"});
        menu.createMenuItem(this.OPTION_FIND_DUPS, {text: this.getMessage("mi_find_duplicates"), image: "TaskNormal"});
        
        // add listeners
        menu.addPopupListener(new AjxListener(this, this.moreActionsMenuListener, [menu, controller]));
        menu.addSelectionListener(this.OPTION_MERGE, new AjxListener(this, this.mergeContactsMenuListener, controller), 0);
        menu.addSelectionListener(this.OPTION_FIND_DUPS, new AjxListener(this, this.openCleanerDialog), 0);
        
        button.setMenu(menu);
    }
};

/**
 * Called when the option menu for "More actions" popsup; disables the 
 * "Merge contacts" option if there are less than 2 items selected
 * @param menu the option menu
 * @param controller the app controller
 */
CcsContactCleaner.prototype.moreActionsMenuListener = function(menu, controller) {
    menu.enable(this.OPTION_MERGE, controller.getCurrentView().getSelectionCount() > 1);
};
/**
 * Called when the option "Merge contacts" is selected; gets the selected items
 * and merges them 
 * @param controller the app controller
 */
CcsContactCleaner.prototype.mergeContactsMenuListener = function(controller) {
    var selectedContacts = controller.getCurrentView().getSelection();
    var numContacts = selectedContacts.length;
    if (numContacts > 0) { 
        var duplicate = new DupContact(0);    
        for ( var i = 0; i < numContacts; i++) {
            duplicate.addContact(selectedContacts[i]);
        }
        this.cleanUp({duplicate:duplicate, merge:true});
    }
};

/**
 * Listener for the close/skip-all button
 * @param ev
 */
CcsContactCleaner.prototype.closeWindowBtnListener = function(ev) {
    this.close();
};

/**
 * Listener for the merge all button
 * @param params
 */
CcsContactCleaner.prototype.mergeAllBtnListener = function(params) {
    this.cleanUp({mergeRemaining:true});
    this.showWaitingDialog();
};

/**
 * Listener for buttons that are shown as actions on duplicates.
 * @param params.action the action keep/merge/skip
 * @param params.dup the dupicate object
 * @param params.index the current duplicate object index
 * 
 */
CcsContactCleaner.prototype.applyActionButtonListener = function(params) {
    switch (params.action) {
        case this.KEEP_ONE:
            this.cleanUp({duplicate:params.dup, contactToKeep:params.contactId});
            break;
        case this.MERGE:
            this.cleanUp({duplicate:params.dup, merge:true});
            break;
        case this.DO_NOTHING:
            break;
        default:
            throw "Invalid action"; 
            break;
    }
    this.nextContact();
};

/**
 * This helper object is a simple iterator to be used for
 * keeping track of the current duplicate being processed.
 * 
 */
function DuplicateIterator(iterable) {
    this.index = 0;
    this.iterable = iterable;
    this.size = iterable.length;
}
/**
 * Returns the next element on the array
 * @returns
 */
DuplicateIterator.prototype.next = function() {
    if (this.index < this.size) {
        return this.iterable[this.index++];
    }
};

/**
 * Return the current element on the list
 * @returns
 */
DuplicateIterator.prototype.current = function() {
    var currentIndex = this.index - 1;
    if (currentIndex < this.size && currentIndex > -1) {
        return this.iterable[currentIndex];
    }
};

/**
 * Returns the previous element.
 * @returns
 */
DuplicateIterator.prototype.previous = function() {
    if (this.index > 0) {
        return this.iterable[--this.index];
    }
};

/**
 * 
 * @returns {Boolean} true if there is a next element.
 */
DuplicateIterator.prototype.hasNext = function() {
    return this.index < this.size;
};

/**
 * Returns the number of item in the array, starting on 1.
 * @returns {Number}
 */
DuplicateIterator.prototype.currentItemNumber = function() {
    return this.index;
};
/**
 * Gets the size of the array.
 * @returns
 */
DuplicateIterator.prototype.getSize = function() {
    return this.size;
};

/**
 * Helper sets
 */
function EmailSet() {
    this.hash = {};
    this.emails = [];
    this.add = function(email) {
        if (!this.hash.hasOwnProperty(email)) {
            this.hash[email] = email;
            this.emails.push(email);
        }
    };
};

function TagSet() {
    this.hash = {};
    this.size = 0;
    this.add = function(tags) {
        if (!tags) {
            return;
        }
        if (typeof tags === "string") {
            tags = tags.split(",");
        }
        var numtags = tags.length;
        for (var i = 0; i < numtags; i++) {
            if (!this.hash.hasOwnProperty(tags[i])) {
                this.hash[tags[i]] = true;
                this.size++;
            }
        }
    };
};


